import pyvisa
import argparse
import time
from datetime import datetime, timezone
from keithley import parse_record
import os

rm = pyvisa.ResourceManager('@py')

def readValue(dev, query_str: str) -> str:
    """Send a query (query_str) to the device resource `dev`, read the response, and return it parsed as float.

    Errors: raises ValueError if parsing fails, pyvisa errors if IO fails.
    """
    dev.write(query_str)
    resp = dev.read().strip()
    print(f"Raw response: {resp!r}")
    return resp

def main():
    parser = argparse.ArgumentParser(description='Very small single-instrument reader')
    parser.add_argument('--pad', type=int, default=14, help='GPIB address of the instrument')
    parser.add_argument('--query', type=str, default='', help='Query string to send to the instrument')
    parser.add_argument('--interval', type=float, default=1.0, help='Seconds between reads')
    parser.add_argument('--display-only', action='store_true', help='Only print values; do not log')
    parser.add_argument('--init-string', type=str, default="N0X;J0X;F0,0X;B5.0,0,0X;L0.1,0X;H0X;G15,0,0X;N1X", help='Optional string (semicolon-separated) to send once to the instrument at startup; use ";" to send multiple commands')
    args = parser.parse_args()

    resource_str = f"GPIB0::{args.pad}::INSTR"
    try:
        dev = rm.open_resource(resource_str)
        intf = rm.open_resource("GPIB0::INTFC")
    except Exception as e:
        print(f"Failed to open resource {resource_str}: {e}")
        return

    print(f"Connected to {resource_str}. Reading '{args.query}' every {args.interval} s. Ctrl+C to stop.")
    try:
        val=""
        dev.clear()
        lib = rm.visalib
        lib.gpib_send_ifc(intf.session)
        dev.read_termination = '\r\n'
        dev.write_termination = '\r\n'
        # If provided, send an initialization string once (do not attempt to read a reply here)
        # Support multiple init commands separated by ';' (e.g. --init-string "a;b;c")
        if args.init_string:
            try:
                parts = [s.strip() for s in args.init_string.split(';') if s.strip()]
                print(f"Sending init string(s): {parts!r}")
                for p in parts:
                    print(f"Sending: {p}")
                    dev.write(p)
                    res = dev.read()
                    print(f"Received: {res}")
                    # small delay between commands to give device time to process
                    time.sleep(.05)
            except Exception as e:
                print(f"Failed to send init string(s): {e}")

        # Prepare CSV file once at startup: if an existing file exists, rotate it by renaming to
        # keithley-<iso timestamp>.csv, then create a fresh keithley.csv with header.
        filename = './keithley.csv'
        if os.path.exists(filename):
            iso_ts = datetime.now(timezone.utc).isoformat()
            safe_ts = iso_ts.replace(':', '').replace('+', '_')
            rotated_name = f"keithley-{safe_ts}.csv"
            try:
                os.rename(filename, rotated_name)
                print(f"Rotated existing '{filename}' -> '{rotated_name}'")
            except Exception as e:
                print(f"Failed to rotate existing CSV '{filename}': {e}")
        try:
            with open(filename, 'w') as fh:
                fh.write('timestamp,voltage,current\n')
        except Exception as e:
            print(f"Failed to create CSV '{filename}': {e}")

        while True:
            ts = datetime.now(timezone.utc).isoformat()
            v=None
            i=None
            try:
                rawRead = readValue(dev, args.query)
                parsed_record = parse_record(rawRead)
                for sub_records in parsed_record:
                    flags, mode, value, meas  = sub_records
                    if (meas == 'V'):
                        v = value
                    elif (meas == 'I'):
                        i = value
                print(f"{ts},{v},{i}")
            except Exception as e:
                print(f"Read failed at {ts}: {e}")
            if args.display_only:
                time.sleep(args.interval)
                continue
            try:
                # Append the latest reading to the pre-created CSV
                with open(filename, 'a') as f:
                    f.write(f"{ts},{v},{i}\n")
            except Exception as e:
                print(f"Failed to write CSV: {e}")
            time.sleep(args.interval)
    except KeyboardInterrupt:
        print('Stopped by user')


if __name__ == '__main__':
    main()
